<?php

namespace backend\modules\fruserdynav2\models;

use yii\web\IdentityInterface;
use Yii;
use backend\modules\refdynawebv2\models\Ref;

/**
 * This is the model class for table "frontend_user".
 *
 * @property int $id
 * @property string $username
 * @property string $auth_key
 * @property string $password_hash
 * @property string $password_reset_token
 * @property string $email
 * @property int $last_login
 * @property int $last_login_attempts
 * @property int $login_attempts
 * @property int $status
 * @property int $role
 * @property string $created_by
 * @property int $created_at
 * @property string $updated_by
 * @property int $updated_at
 */
class FrontendUser extends \yii\db\ActiveRecord implements IdentityInterface {

    CONST STATUS_ACTIVE = '1';

    public $password;

    /**
     * {@inheritdoc}
     */
    public static function tableName() {
        return 'frontend_user';
    }

    /**
     * {@inheritdoc}
     */
    public function rules() {
        return [
            [['last_login', 'last_login_attempts', 'login_attempts'], 'integer'],
            [['username', 'auth_key', 'password_hash', 'password_reset_token', 'email'], 'string', 'max' => 255],
            [['created_by', 'updated_by', 'created_at', 'updated_at', 'remarks', 'img_url',
            'created_user_type', 'updated_user_type','password'], 'safe'],
            [['designation', 'department'], 'string', 'max' => 60],
            [['username', 'password_hash', 'status', 'role', 'fullname', 'email', 'mobile_no', 'office_no', 'designation', 'gender',
            'service_status', 'service_status_latest', 'service_group', 'service_classification', 'gred', 'gred_post',
            'state', 'branch', 'department', 'unit',], 'required',
                'message' => Yii::t('app', '{attribute} cannot be blank')],
            [['email'], 'email', 'message' => Yii::t('app', '{attribute} is not a valid email address')],
            [['username', 'email'], 'unique', 'message' => Yii::t('app', '{attribute} already exists')],
            ['password', 'required', 'on' => 'create'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels() {
        return [
            'id' => Yii::t('app', 'ID'),
            'username' => Yii::t('app', 'Username (IC No.)'),
            'fullname' => Yii::t('app', 'Name'),
            'confirmpassword' => Yii::t('app', 'Confirm Password'),
            'auth_key' => Yii::t('app', 'Auth Key'),
            'password_hash' => Yii::t('app', 'Password Hash'),
            'password_reset_token' => Yii::t('app', 'Password Reset Token'),
            'email' => Yii::t('app', 'Email'),
            'last_login' => Yii::t('app', 'Last Login'),
            'last_login_attempts' => Yii::t('app', 'Last Login Attempts'),
            'login_attempts' => Yii::t('app', 'Login Attempts'),
            'status' => Yii::t('app', 'Status'),
            'role' => Yii::t('app', 'User Category'),
            'designation' => Yii::t('app', 'Designation'),
            'created_by' => Yii::t('app', 'Created By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'updated_at' => Yii::t('app', 'Updated At'),
            'remarks' => Yii::t('app', 'Remarks'),
            'mobile_no' => Yii::t('app', 'Mobile Number'),
            'office_no' => Yii::t('app', 'Office Number'),
            'gender' => Yii::t('app', 'Gender'),
            'service_status' => Yii::t('app', 'Service Status'),
            'service_status_latest' => Yii::t('app', 'User Status'),
            'service_group' => Yii::t('app', 'Service Group'),
            'service_classification' => Yii::t('app', 'Service Classification'),
            'gred' => Yii::t('app', 'Grade'),
            'gred_post' => Yii::t('app', 'Designation / Grade'),
            'state' => Yii::t('app', 'State'),
            'branch' => Yii::t('app', 'Branch'),
            'department' => Yii::t('app', 'Department'),
            'unit' => Yii::t('app', 'Unit'),
            'img_url' => Yii::t('app', 'User Image'),
        ];
    }

    /**
     * @inheritdoc
     */
    public static function findIdentity($id) {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null) {
        throw new NotSupportedException('"findIdentityByAccessToken" is not implemented.');
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username) {
        return static::findOne(['username' => $username, 'status' => self::STATUS_ACTIVE]);
    }

    public function getStaff() {
        return $this->hasOne(\backend\models\Staff::className(), ['userid' => 'id']);
    }

    /**
     * Finds user by password reset token
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByPasswordResetToken($token) {
        if (!static::isPasswordResetTokenValid($token)) {
            return null;
        }

        return static::findOne([
                    'password_reset_token' => $token,
                    'status' => self::STATUS_ACTIVE,
        ]);
    }

    /**
     * Finds out if password reset token is valid
     *
     * @param string $token password reset token
     * @return boolean
     */
    public static function isPasswordResetTokenValid($token) {
        if (empty($token)) {
            return false;
        }

        $timestamp = (int) substr($token, strrpos($token, '_') + 1);
        $expire = Yii::$app->params['user.passwordResetTokenExpire'];
        return $timestamp + $expire >= time();
    }

    /**
     * @inheritdoc
     */
    public function getId() {
        return $this->getPrimaryKey();
    }

    /**
     * @inheritdoc
     */
    public function getRoles() {
        // $tmp = [];
        // $roles = RolesAssigned::findAll(['staff_info_id'=>$this->id]);
        // if ($roles !== null){
        //     foreach ($roles as $r){
        //         $tmp[] = $r->role_code;
        //     }
        // }
        // return $tmp;

        return $this->role;
    }

    /* custom */

    public function getStaffRoles() {
        $tmp = [];
        $roles = RolesAssigned::findAll(['staff_info_id' => Yii::$app->user->identity->staff->id]);
        if ($roles !== null) {
            foreach ($roles as $r) {
                $tmp[] = $r->role_code;
            }
        }
        return $tmp;
    }

    /**
     * @inheritdoc
     */
    public function getLastlogin() {
        return date('j F Y g:i A', $this->last_login);
    }

    /**
     * @inheritdoc
     */
    public function getAuthKey() {
        //return $this->auth_key;
        return true;
    }

    /**
     * @inheritdoc
     */
    public function getFullname() {
        $profile = Profile::find()->where(['userid' => $this->id])->one();
        if ($profile !== null)
            return $profile->staff_name;

        return false;
    }

    /**
     * @inheritdoc
     */
    public function validateAuthKey($authKey) {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return boolean if password provided is valid for current user
     */
    public function validatePassword($password) {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function setPassword($password) {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    /**
     * Generates "remember me" authentication key
     */
    public function generateAuthKey() {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Generates new password reset token
     */
    public function generatePasswordResetToken() {
        $this->password_reset_token = Yii::$app->security->generateRandomString() . '_' . time();
    }

    /**
     * Removes password reset token
     */
    public function removePasswordResetToken() {
        $this->password_reset_token = null;
    }

    public static function getFrontendUser($sort = 'fullname') {
        $arr2 = [];

        $arr = self::find()->orderBy($sort)->all();

        foreach ($arr as $user) {
            $arr2[$user->id] = $user->fullname;
        }
        return $arr2;
    }

    public static function getFrontendUSerFromID($id) {
        $returnval = '- Tiada -';

        if ($id > 0) {

            $arr = self::find()->where(['id' => $id])->all();

            foreach ($arr as $user) {
                $returnval = $user->fullname;
            }
        }
        return $returnval;
    }

    public static function getFrontendUserFromOneID($id) {
        $arr = self::find()->where(['id' => $id])->one();

        return empty($arr) ? '' : $arr['fullname'];
    }

    public static function getname($id) {
        $output = '';
        if (($model = self::findOne($id)) !== null)
            $output = $model->fullname;
        return $output;
    }

    public static function getparentid($code, $cat) {
        $output = 0;
        if (($model = Ref::findOne(['code' => $code, 'cat' => $cat])) !== null)
            $output = $model->id;
        return $output;
    }

}
